"use client";

import React, { useEffect, useState } from "react";
import InnerBanner from "../../../components/InnerBanner";
import "../page.css";
import Link from "next/link";
import { useCart } from "@/context/CartContext";

function PageLoader() {
    return (
        <div className="min-vh-100 d-flex align-items-center justify-content-center">
            <div className="spinner-border text-primary" />
        </div>
    );
}

function stripHtml(html: string): string {
    const div = document.createElement("div");
    div.innerHTML = html;
    return div.textContent || div.innerText || "";
}

type SelectionState = {
    type?: string;
    medium?: string;
    session?: string;
};

interface GuessCoursesClientProps {
    courseCode: string;
    headline: string;
    description: string;
    breadcrumbHeadline: string;
    allAssignments: any[];
}

export default function GuessCoursesClient({
    courseCode,
    headline,
    description,
    breadcrumbHeadline,
    allAssignments
}: GuessCoursesClientProps) {
    const [mounted, setMounted] = useState(false);
    const [search, setSearch] = useState("");

    useEffect(() => {
        // Load Bootstrap and FontAwesome dynamically if not present
        if (!document.querySelector('link[href*="bootstrap.min.css"]')) {
            const bs = document.createElement("link");
            bs.rel = "stylesheet";
            bs.href = "https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css"; // Added actual URL or placeholder
            document.head.appendChild(bs);
        }
        if (!document.querySelector('link[href*="font-awesome"]')) {
            const fa = document.createElement("link");
            fa.rel = "stylesheet";
            fa.href = "https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css"; // Added actual URL
            document.head.appendChild(fa);
        }
        setTimeout(() => setMounted(true), 100);
    }, []);

    const { addToCart } = useCart();

    // Derived state for description stripped of HTML (though we used dangerSetInnerHTML in original, stripHtml might be used for meta description or fallback)
    // The original code used stripHtml for description state, but rendered it with dangerouslySetInnerHTML. 
    // Wait, in original code:
    // setDescription(stripHtml(data.course.description));
    // <p dangerouslySetInnerHTML={{ __html: description }} />
    // If we strip HTML, then dangerouslySetInnerHTML is redundant but safe.
    // However, if the description had formatting, it's lost.
    // Let's keep it as passed from server, assuming server passes raw HTML for display, or stripped if intended.
    // The server fetch in original code: `setDescription(stripHtml(data.course.description))`
    // So it was stripping HTML effectively.

    const [selections, setSelections] = useState<Record<string, SelectionState>>({});
    const [activeCartSlug, setActiveCartSlug] = useState<string | null>(null);

    const updateSelection = (
        slug: string,
        field: keyof SelectionState,
        value: string
    ) => {
        setSelections(prev => ({
            ...prev,
            [slug]: {
                ...prev[slug],
                [field]: value,
            },
        }));
    };

    const handleAddToCart = async (item: any) => {
        const selection = selections[item.guess_paper_slug];
        if (!selection) return;

        setActiveCartSlug(item.guess_paper_slug);

        const rawString = `${courseCode}|${item.guess_paper_slug}|${Date.now()}`;

        try {
            await addToCart({
                id: `guess-paper-${rawString}`,
                name: item.title ?? "",
                price: item.sale ?? 0,
                qty: 1,
                img: item.img ?? "",
                courseCode,
                itemCode: item.guess_paper_slug,
                subjectIds: null,
                selectedType: null,
                selectedMedium: selection.medium!,
                selectedSession: null,
                dataSaveId: null,
                pageName: "guess-paper-page",
            });
        } finally {
            setActiveCartSlug(null);
        }
    };

    const filteredAssignments = allAssignments.filter(a =>
        a.title.toLowerCase().includes(search.toLowerCase()) ||
        a.code?.toLowerCase().includes(search.toLowerCase())
    );

    return (
        <>
            <InnerBanner title={headline} breadcrumb={breadcrumbHeadline} />

            <div className="page-container hello">
                <div className="container">
                    <div className="d-lg-flex d-md-block d-block align-items-center gap-2 mb-4 justify-content-between">
                        <div className='d-flex '>
                            <h2 className="fw-bold">
                                {headline}
                            </h2>
                            {/* Displaying description. Original code stripped HTML then rendered it as HTML. */}
                            <p dangerouslySetInnerHTML={{ __html: description }} />
                        </div>

                        {/* 🔥 Search Box */}
                        <div className="search-box mb-4 w-50 position-relative">
                            <input
                                type="text"
                                className="form-control "
                                placeholder="Search by Title or Guess Paper Code..."
                                value={search}
                                onChange={(e) => setSearch(e.target.value)}
                            />

                            <i className="fa-solid fa-magnifying-glass"></i>
                        </div>
                    </div>

                    <div className="row g-4">
                        {filteredAssignments.length > 0 ? (
                            filteredAssignments.map((item: any, i: number) => {
                                const discount = Math.round(((item.price - item.sale) / item.price) * 100);

                                const currentSelection = selections[item.guess_paper_slug];
                                const isDisabled =
                                    !currentSelection?.medium ||
                                    activeCartSlug === item.guess_paper_slug;

                                return (
                                    <div key={i} className="col-12 col-sm-6 col-md-6 col-lg-4 col-xl-3">
                                        <div className="card h-100 border-0 shadow-sm product-card">
                                            <div className="position-relative overflow-hidden">
                                                <span className="badge bg-danger position-absolute top-0 end-0 m-3 z-2">
                                                    Sale!
                                                </span>
                                                <div className="product-image-wrapper m-0">
                                                    <img
                                                        src={item.img}
                                                        className="card-img-top product-image"
                                                        alt={item.title}
                                                    />
                                                </div>
                                            </div>

                                            <div className="card-body p-3">
                                                <Link href={`/guess-papers/${courseCode}/${item.guess_paper_slug}`} className="text-decoration-none">
                                                    <h5 className="card-title text-primary fw-bold mb-1">{item.code}</h5>
                                                </Link>
                                                <p className="card-text text-muted small mb-3 text-truncate" title={item.title}>
                                                    {item.title}
                                                </p>

                                                <div className="mb-3">
                                                    {/* Languages */}
                                                    <select
                                                        className="form-select form-select-sm mb-2 shadow-none border-secondary-subtle"
                                                        value={currentSelection?.medium || ""}
                                                        onChange={e =>
                                                            updateSelection(item.guess_paper_slug, "medium", e.target.value)
                                                        }
                                                        aria-label="Language Selection"
                                                    >
                                                        <option value="">-- Select Medium --</option>
                                                        {item.medium?.map((language: { enc_id: string; language_name: string }) => (
                                                            <option key={language.enc_id} value={language.enc_id}>{language.language_name}</option>
                                                        ))}
                                                    </select>
                                                </div>

                                                <div className="d-flex align-items-center mb-3">
                                                    <h4 className="mb-0 fw-bold">₹{item.sale}</h4>
                                                    <span className="text-muted ms-2 text-decoration-line-through small">Only</span>
                                                    <span className="text-muted ms-1 text-decoration-line-through small">MRP ₹{item.price}</span>
                                                    <span className="badge bg-danger ms-auto">{discount}% Off</span>
                                                </div>

                                                <small
                                                    className="text-danger mb-1 d-block"
                                                    style={{ fontSize: "0.72rem", fontStyle: "italic" }}
                                                >
                                                    ** Medium is mandatory.
                                                </small>

                                                <div className="d-grid gap-2">
                                                    <button
                                                        className="btn btn-dark py-2 rounded-3"
                                                        disabled={isDisabled}
                                                        onClick={() => handleAddToCart(item)}
                                                    >
                                                        {activeCartSlug === item.guess_paper_slug ? (
                                                            <>
                                                                <span className="spinner-border spinner-border-sm" /> Adding...
                                                            </>
                                                        ) : (
                                                            "Add to Cart"
                                                        )}
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                );
                            })
                        ) : (
                            <div className="col-12">
                                <div className="empty-state">
                                    <div className="empty-icon">
                                        <i className="fa-solid fa-folder-open"></i>
                                    </div>
                                    <h3 className="mb-3">No Guess Papers Available</h3>
                                    <p className="text-muted">
                                        We're currently preparing guess papers for this category. Please check back soon!
                                    </p>
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </>
    );
}
